import { Pane } from 'https://cdn.skypack.dev/tweakpane@4.0.4';
import gsap from 'https://cdn.skypack.dev/gsap@3.12.0';
import ScrollTrigger from 'https://cdn.skypack.dev/gsap@3.12.0/ScrollTrigger';

const config = {
  theme: 'dark',
  animate: true,
  snap: true,
  start: gsap.utils.random(0, 100, 1),
  end: gsap.utils.random(900, 1000, 1),
  scroll: true,
  debug: false
};

// Config panelini gizle
const ctrl = new Pane({
  title: 'Config',
  expanded: false,
  hidden: true
});

let items;
let scrollerScrub;
let dimmerScrub;
let chromaEntry;
let chromaExit;

const update = () => {
  document.documentElement.dataset.theme = config.theme;
  document.documentElement.dataset.syncScrollbar = config.scroll;
  document.documentElement.dataset.animate = config.animate;
  document.documentElement.dataset.snap = config.snap;
  document.documentElement.dataset.debug = config.debug;
  document.documentElement.style.setProperty('--start', config.start);
  document.documentElement.style.setProperty('--hue', config.start);
  document.documentElement.style.setProperty('--end', config.end);

  if (!config.animate) {
    if (chromaEntry?.scrollTrigger) chromaEntry.scrollTrigger.disable(true, false);
    if (chromaExit?.scrollTrigger) chromaExit.scrollTrigger.disable(true, false);
    if (dimmerScrub) dimmerScrub.disable(true, false);
    if (scrollerScrub) scrollerScrub.disable(true, false);
    if (items) gsap.set(items, { opacity: 1 });
    gsap.set(document.documentElement, { '--chroma': 0 });
  } else if (items && dimmerScrub && scrollerScrub && chromaEntry && chromaExit) {
    gsap.set(items, { opacity: i => i !== 0 ? 0.2 : 1 });
    dimmerScrub.enable(true, true);
    scrollerScrub.enable(true, true);
    chromaEntry.scrollTrigger.enable(true, true);
    chromaExit.scrollTrigger.enable(true, true);
  }
};

const sync = event => {
  if (!document.startViewTransition || event.target.controller.view.labelElement.innerText !== 'Theme') {
    return update();
  }
  document.startViewTransition(() => update());
};

ctrl.addBinding(config, 'animate', { label: 'Animate' });
ctrl.addBinding(config, 'snap', { label: 'Snap' });
ctrl.addBinding(config, 'start', { label: 'Hue Start', min: 0, max: 1000, step: 1 });
ctrl.addBinding(config, 'end', { label: 'Hue End', min: 0, max: 1000, step: 1 });
ctrl.addBinding(config, 'scroll', { label: 'Scrollbar' });
ctrl.addBinding(config, 'debug', { label: 'Debug' });
ctrl.addBinding(config, 'theme', {
  label: 'Theme',
  options: {
    System: 'system',
    Light: 'light',
    Dark: 'dark'
  }
});

ctrl.on('change', sync);

const initializeAnimations = () => {
  if (!CSS.supports('(animation-timeline: scroll()) and (animation-range: 0% 100%)')) {
    gsap.registerPlugin(ScrollTrigger);

    items = gsap.utils.toArray('.scroll-animation-section ul li');

    if (items.length === 0) return; // Eğer elementler bulunamazsa çık

    gsap.set(items, { opacity: i => i !== 0 ? 0.2 : 1 });

    const dimmer = gsap.timeline()
      .to(items.slice(1), {
        opacity: 1,
        stagger: 0.5
      })
      .to(items.slice(0, items.length - 1), {
        opacity: 0.2,
        stagger: 0.5
      }, 0);

    dimmerScrub = ScrollTrigger.create({
      trigger: items[0],
      endTrigger: items[items.length - 1],
      start: 'center center',
      end: 'center center',
      animation: dimmer,
      scrub: 0.2
    });

    const scroller = gsap.timeline().fromTo(
      document.documentElement,
      { '--hue': config.start },
      { '--hue': config.end, ease: 'none' }
    );

    scrollerScrub = ScrollTrigger.create({
      trigger: items[0],
      endTrigger: items[items.length - 1],
      start: 'center center',
      end: 'center center',
      animation: scroller,
      scrub: 0.2
    });

    chromaEntry = gsap.fromTo(
      document.documentElement,
      { '--chroma': 0 },
      {
        '--chroma': 0.3,
        ease: 'none',
        scrollTrigger: {
          scrub: 0.2,
          trigger: items[0],
          start: 'center center+=40',
          end: 'center center'
        }
      }
    );

    chromaExit = gsap.fromTo(
      document.documentElement,
      { '--chroma': 0.3 },
      {
        '--chroma': 0,
        ease: 'none',
        scrollTrigger: {
          scrub: 0.2,
          trigger: items[items.length - 2],
          start: 'center center',
          end: 'center center-=40'
        }
      }
    );

    // Animasyonları başlat
    update();
  }
};

// DOM yüklendiğinde animasyonları başlat
document.addEventListener('DOMContentLoaded', initializeAnimations); 